-- JSON Path Zerlegung
CREATE OR REPLACE FUNCTION x_10_interfaces.jsonb__by_dynamic_json_path__get(
      IN _json_data  jsonb,
      IN _path       varchar
      )
    RETURNS jsonb AS $$
    BEGIN

      IF _path ~ '^\\d+$' THEN
        -- JSON-Array-Element
        _json_data := _json_data -> _path::int;
      ELSEIF _path LIKE '$%' THEN
        -- JSON-Path-Query
        _json_data := jsonb_path_query( _json_data, _path::JSONPATH );
      ELSE
        -- JSON-Objekt-Eigenschaft
        _json_data := _json_data -> _path;
      END IF;

      RETURN _json_data;

    END $$ LANGUAGE plpgsql;
  --

-- JSON Path Zerlegung für Arrays
CREATE OR REPLACE FUNCTION x_10_interfaces.jsonb__by_dynamic_json_path__get(
    IN _json_data  jsonb,
    IN _paths      varchar[]
    )
    RETURNS jsonb AS $$
    DECLARE
      _path  varchar;
    BEGIN

      FOREACH _path IN ARRAY _paths
      LOOP
        IF _path ~ '^\\d+$' THEN
          -- JSON-Array-Element
          _json_data := _json_data -> _path::int;
        ELSEIF _path LIKE '$%' THEN
        -- JSON-Path-Query
        _json_data := jsonb_path_query( _json_data, _path::JSONPATH );
      ELSE
          -- JSON-Objekt-Eigenschaft
          _json_data := _json_data -> _path;
        END IF;
      END LOOP;

      RETURN _json_data;

    END $$ LANGUAGE plpgsql;
  --

CREATE OR REPLACE FUNCTION x_10_interfaces.string__replace_var__by_jsonb_key(
      IN     _json   jsonb,
      INOUT  _text   varchar
    )
    RETURNS varchar AS $$
    DECLARE
      _r    record;
      _str  text;
    BEGIN

      -- Es wird in einem Text nach Variablen gesucht die mit <#...#> geklammert sind.
      -- Diese werden als als JSON-Pfad interpretiert.
      -- Dieser JSON-Pfad wird im mitgelierten JSON-Objekt aufgelöst.
      -- Anschließend wird der Text mit den aufgelösten Variablen wieder ausgegeben.

      _str := '';

      FOR _r IN
        SELECT "unnest" AS text_part,                                                                                       -- Textbestandteil
                coalesce(x_10_interfaces.jsonb__by_dynamic_json_path__get( _json, "unnest" ) #>> '{}') AS var_text,         -- übersetze Variable
                --x_10_interfaces.jsonb__by_dynamic_json_path__get( _json, "unnest" ) #>> '{}' AS var_text,                 -- übersetze Variable
                MOD( ROW_NUMBER() OVER( ORDER BY (SELECT 1) ), 2 ) = 0 AS is_var                                            -- gerade Zeile sind Variablen
           FROM unnest( regexp_split_to_array( _text, '<#|#>' ) )
      LOOP
        IF _r.is_var AND _r.var_text is null THEN
          RAISE NOTICE 'Wert für <#%s#> wurde nicht gefunden', _r.text_part;
          _str := null;
          continue;
        ELSE
          _str := _str || CASE WHEN _r.is_var THEN _r.var_text ELSE _r.text_part END;
        END IF;

      END LOOP;

      _text := coalesce(_str,'null');

    END $$ LANGUAGE plpgsql;

-- Funktion schreibt Anfrage
CREATE OR REPLACE FUNCTION x_10_interfaces.nexar__import_anfangebot__by_jsonb(
      IN _sellers   jsonb,
      IN _aknr      varchar
      )
    RETURNS VOID AS $$
    DECLARE
      _element      record;
      _ad_krz       varchar;
      _anf_nr       varchar;
      _aart_id      integer;
      _alief_id     integer;
      _aang_id      integer;
      _AnfPos       integer;
    BEGIN

      -- Standard-Anfragenummer unter der Nexar-Angebote abgelegt werden definieren
      _anf_nr := coalesce( nullif( tsystem.settings__gettext( 'NEXAR_ANF_NR' ), '' ), 'NEXAR' );

      -- Anfrage anlegen falls noch nicht vorhanden
      IF NOT EXISTS( SELECT _anf_nr FROM anfrage WHERE anf_nr = _anf_nr ) THEN
        INSERT INTO anfrage ( anf_nr  )
            VALUES       ( _anf_nr );
      END IF;

      -- Anfrageartikel finden
      _aart_id = ( SELECT aart_id
              FROM anfart
              WHERE aart_anf_nr = _anf_nr
                AND aart_ak_nr  = _aknr
              LIMIT 1);

      -- Falls noch nicht vorhanden wird der Artikel in die Anfrage aufgenommen
      IF _aart_id is null THEN
        -- Nummer für Anfrageposition ermitteln
        _AnfPos := coalesce( ( SELECT max( aart_pos ) FROM anfart WHERE aart_anf_nr = _anf_nr ), 0 ) + 1;
        -- Anfrageartikel anlegen
        -- Pflichtfelder schreiben
        INSERT INTO anfart ( aart_pos, aart_ak_nr, aart_anf_nr, aart_menge, aart_m_id )
               VALUES      ( _AnfPos , _aknr     , _anf_nr    , 1         , 1         )
               RETURNING aart_id INTO _aart_id;

        -- dynamische Felder schreiben
        PERFORM x_10_interfaces.nexar__import_dynamic_fields__by_jsonb(_sellers, 'anfart', 'aart_id', _aart_id::varchar);

      END IF;

      -- Alle Verkäufer durchlaufen
      FOR _element IN
        SELECT jsonb_array_elements FROM jsonb_array_elements(_sellers)
      LOOP

        -- Debug-Info
        RAISE NOTICE 'Lieferanten-ID: %', x_10_interfaces.jsonb__by_dynamic_json_path__get(_element.jsonb_array_elements, '$.company' ) ->> 'id';

        -- Lieferant in Stammdaten finden
        SELECT ad_krz
          INTO _ad_krz
          FROM adk
         WHERE ad_nexar_id = x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.company' ) ->> 'id';

        -- Wenn es keinen gibt ~> nächsten suchen
        CONTINUE WHEN _ad_krz is null;

        -- Prüfen ob Lieferant bereits in Liste der angefragten Lieferanten enthalten ist
        _alief_id = ( SELECT alief_id
                        FROM anflief
                       WHERE alief_anf_nr = _anf_nr
                         AND alief_lkn    = _ad_krz );

        -- Falls noch nicht vorhanden wird der Lieferant in die Anfrage aufgenommen
        IF _alief_id is null THEN
          -- Pflichtfelder schreiben
          INSERT INTO anflief ( alief_anf_nr, alief_lkn )
                 VALUES       ( _anf_nr     , _ad_krz   )
                 RETURNING alief_id INTO _alief_id;
          -- dynamische Felder schreiben
          PERFORM x_10_interfaces.nexar__import_dynamic_fields__by_jsonb( x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.company' ), 'anflief', 'alief_id', _alief_id::varchar );
        END IF;

        -- altes Angebot löschen
        DELETE FROM anfangebot
              WHERE aang_aart_id  = _aart_id
                AND aang_alief_id = _alief_id;

        -- Debug Info
        RAISE NOTICE 'Lieferanten-Artikelnummer: %', x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.offers[0]' ) ->> 'sku';

        -- Wenn kein Lagerbestand vorhanden und Setting aktiv ~> Preis nicht anlegen
        CONTINUE WHEN x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.offers[0]' ) ->> 'inventoryLevel' <= 0 AND tsystem.Settings__GetBool('NEXAR_skip_lkn_lg_anztot_leer');

        -- Lieferantenpreis schreiben
        -- Pflichtfelder schreiben
        INSERT INTO anfangebot ( aang_aart_id,  aang_alief_id,
                                 aang_akref,
                                 aang_lieferzeit,
                                 aang_lkn_lg_anztot,
                                 aang_los
                                )
               VALUES          ( _aart_id    , _alief_id,
                                 x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.offers[0]' ) ->> 'sku',
                                 -- Wenn Lagerbestand vorhanden dann Lieferzeit = 0 Tage, sonst ab Werk Lieferzeit
                                 CASE WHEN x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.offers[0]' ) ->> 'inventoryLevel' > 0
                                   THEN 0
                                   ELSE coalesce( ( x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.offers[0]' ) ->> 'factoryLeadDays' )::integer, 0 ) END,
                                 --  Lagerbestand als Angebotsmenge
                                 max( coalesce( ( x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.offers[0]' ) ->> 'inventoryLevel' ), '0' )::integer, 0 ),
                                 -- Mindestbestellmenge als Losgröße
                                 ( x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.offers[0]' ) ->> 'moq' )::integer
                                )
               RETURNING aang_id INTO _aang_id;

        -- dynamische Felder schreiben
        PERFORM x_10_interfaces.nexar__import_dynamic_fields__by_jsonb( x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.offers[0]' ), 'anfangebot', 'aang_id', _aang_id::varchar);

        -- Staffeln schreiben
        EXECUTE x_10_interfaces.nexar__import_epreisstaffel__by_jsonb( x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.offers[0].prices' ), null, null, _aang_id );

      END LOOP;

    END $$ LANGUAGE plpgsql;
  --

-- Funktion schreibt Lieferantenpreise
CREATE OR REPLACE FUNCTION x_10_interfaces.nexar__import_epreis__by_jsonb(
      IN _sellers         jsonb,
      IN _e_aknr          varchar
      )
    RETURNS VOID AS $$
    DECLARE
      _element      record;
      _ad_krz       varchar;
      _e_id         integer;
      _offer        record;
    BEGIN

      -- Alle Verkäufer durchlaufen
      FOR _element IN
          SELECT jsonb_array_elements FROM jsonb_array_elements(_sellers)
      LOOP

        -- Debug-Info
        RAISE NOTICE 'Lieferanten-ID: %', x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.company' ) ->> 'id';

        -- Lieferant in Stammdaten finden
        SELECT ad_krz
          INTO _ad_krz
          FROM adk
        WHERE ad_nexar_id = x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.company' ) ->> 'id';

        -- Wenn es keinen passenden Prodat-Lieferanten gibt ~> nächsten suchen
        CONTINUE WHEN _ad_krz is null;

        -- alten Nexar-Preis löschen
        DELETE FROM epreis
              WHERE e_aknr     = _e_aknr
                AND e_lkn      = _ad_krz
                AND e_herkunft = 'API-Nexar';

        -- Alle Angebote durchlaufen
        FOR _offer IN
            SELECT jsonb_array_elements from jsonb_array_elements(x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.offers' ))
        LOOP

          -- Debug-Info
          RAISE NOTICE 'Lieferanten-Artikelnummer: %', _offer.jsonb_array_elements ->> 'sku';

          -- Wenn kein Lagerbestand vorhanden und Setting aktiv ~> Preis nicht anlegen
          CONTINUE WHEN _offer.jsonb_array_elements ->> 'inventoryLevel' <= 0 AND tsystem.Settings__GetBool('NEXAR_skip_lkn_lg_anztot_leer');

          -- Pflichtfelder schreiben
          -- Lieferantenpreis schreiben
          INSERT INTO epreis (e_aknr,  e_lkn,
                              e_best,
                              e_herkunft,
                              e_lfzt)
                VALUES      (_e_aknr, _ad_krz,
                              _offer.jsonb_array_elements ->> 'sku',
                              'API-Nexar',
                              coalesce( ( _offer.jsonb_array_elements ->> 'factoryLeadDays' )::integer, 0 ) )
                RETURNING e_id INTO _e_id;

          -- dynamische Felder schreiben
          PERFORM x_10_interfaces.nexar__import_dynamic_fields__by_jsonb( _offer.jsonb_array_elements, 'epreis', 'e_id', _e_id::varchar );

          -- Staffeln schreiben
          EXECUTE x_10_interfaces.nexar__import_epreisstaffel__by_jsonb( x_10_interfaces.jsonb__by_dynamic_json_path__get( _offer.jsonb_array_elements, '$.prices' ), _e_id );

        END LOOP;

      END LOOP;

    END $$ LANGUAGE plpgsql;
  --

-- Funktion schreibt Staffeln
CREATE OR REPLACE FUNCTION x_10_interfaces.nexar__import_epreisstaffel__by_jsonb(
      IN _prices          jsonb,
      IN _est_e_id        integer  DEFAULT null,
      IN _est_aart_id     integer  DEFAULT null,
      IN _est_aang_id     integer  DEFAULT null
      )
    RETURNS VOID AS $$
    DECLARE
      _r            record;
      _element      record;
      _est_id       integer;
    BEGIN

      IF _est_e_id is null AND _est_aart_id is null AND _est_aang_id is null THEN
        RAISE EXCEPTION 'Staffelpreise können nicht ohne Bezug angelegt werden';
      END IF;

      -- Alte Staffel löschen
      EXECUTE 'DELETE FROM epreisstaffel WHERE ' || coalesce( _est_e_id::varchar, 'null' ) || ' = est_e_id OR ' || coalesce( _est_aart_id::varchar, 'null' ) || ' = est_aart_id OR ' || coalesce( _est_aang_id::varchar, 'null' ) || ' = est_aang_id';

      FOR _element IN
          SELECT jsonb_array_elements FROM jsonb_array_elements(_prices)
      LOOP

        -- Staffelpreise schreiben
        -- Pflichtfelder schreiben

          INSERT INTO epreisstaffel ( est_e_id,
                                      est_aart_id,
                                      est_aang_id,
                                      est_ep,
                                      est_mengevon
                                    )
                  VALUES            ( _est_e_id,
                                      _est_aart_id,
                                      _est_aang_id,
                                      ( _element.jsonb_array_elements ->> 'convertedPrice' )::numeric,
                                      ( _element.jsonb_array_elements ->> 'quantity' )::numeric
                                    )
                  RETURNING est_id INTO _est_id;

        -- dynamische Felder schreiben
        PERFORM x_10_interfaces.nexar__import_dynamic_fields__by_jsonb( _element.jsonb_array_elements, 'epreisstaffel', 'est_id', _est_id::varchar);

      END LOOP;

    END $$ LANGUAGE plpgsql;
  --


-- Funktion ruft Unterfunktionen zum JSON-Import auf
CREATE OR REPLACE FUNCTION x_10_interfaces.interfaces__import_data__by_ii_id(
      IN _ii_id             integer,
      IN _write_to_art      boolean DEFAULT false,
      IN _write_to_epreis   boolean DEFAULT false,
      IN _write_to_anfrage  boolean DEFAULT true
      )
    RETURNS VOID AS $$
    DECLARE
      _r            record;
      _ak_nr        varchar;
      _bvp_bvsnr    varchar;
    BEGIN

      SELECT *
        INTO _r
        FROM x_10_interfaces.interfaces_import
      WHERE ii_id = _ii_id;

      -- Nexar
      IF _r.ii_connector = 'Nexar' THEN

        -- versuchen Artikelnummer aus den Variablen zu holen
        _ak_nr = _r.ii_ownerform_var ->> 'ak_nr';

        IF _ak_nr is not null THEN
          -- JSON beinhaltet Daten zu einem einzelnen Artikel
          PERFORM x_10_interfaces.nexar__import__from_jsonb( _r.ii_result::jsonb, _ak_nr, _write_to_art, _write_to_epreis, _write_to_anfrage );
        END IF;

        -- versuchen Bestellvorschlagsnummer aus den Variablen zu holen
        _bvp_bvsnr = _r.ii_ownerform_var ->> 'bvp_bvsnr';

        IF _bvp_bvsnr is not null THEN
          -- JSON beinhaltet Daten zu Artikeln aus einem Bestellvorschlag
          PERFORM x_10_interfaces.nexar__import_multi__from_jsonb( _r.ii_result::jsonb, _write_to_art, _write_to_epreis, _write_to_anfrage );
        END IF;

      END IF;

    END $$ LANGUAGE plpgsql;
  --

-- Funktion schreibt dynamische Felder
CREATE OR REPLACE FUNCTION x_10_interfaces.nexar__import_dynamic_fields__by_jsonb(
      IN _json   jsonb,
      IN _table  varchar,
      IN _field  varchar,
      IN _key    varchar
    )
    RETURNS VOID AS $$
    DECLARE
      _r         record;
      _recno     record;
      _r_dbrid   integer;
      _value     varchar;
      _where     varchar;
      _query     varchar;
    BEGIN

      -- alle Felder aus den dynamischen Settings durchlaufen
      FOR _r IN (
        SELECT nexar_mapping.*
          FROM x_10_interfaces.nexar_mapping
        WHERE nexm_table = _table::regclass
        )
      LOOP

        IF _r.nexm_mapping_typ = 'Eingabefeld' THEN
        -- Eingabefelder schreiben

          -- Wert für Feld aus JSON gemäß Setting-Tabelle ermitteln
          _value = x_10_interfaces.string__replace_var__by_jsonb_key( _json, _r.nexm_txt );

          -- Wenn kein Wert gefunden wurde ~> nichts schreiben
          continue WHEN _value IS null
                     OR _value = 'null'
                     OR _value = '';

          -- Werte für Zielfeld casten
          --_value = 'CAST(''' || _value || ''' as ' || coalesce(_r.nexm_cast, 'varchar') || ')';
          _value = format( 'CAST( %s as %s)', _value, coalesce( _r.nexm_cast, 'varchar' ) );


          -- Standard Where-Teil definieren
          _where = ' WHERE ' || _field || ' = ''' || _key || '''';

          -- zusätzlichen Where-Teil ergänzen falls definiert
          IF _r.nexm_where IS NOT null AND _r.nexm_where <> '' THEN
            _where = _where || ' AND ' || _r.nexm_where;
          END IF;

          -- Update Statement zusammenbauen und ausführen
          _query := format( 'UPDATE %I SET %I = %s %s', _r.nexm_table::varchar, _r.nexm_column::varchar, _value, _where);
          EXECUTE _query;

        ELSEIF _r.nexm_mapping_typ = 'Parameter' THEN
        -- Parameter schreiben

          -- Wert für Feld aus JSON gemäß Setting-Tabelle ermitteln
          _value = x_10_interfaces.string__replace_var__by_jsonb_key( _json, _r.nexm_txt );

          -- Wenn kein Wert gefunden wurde ~> nichts schreiben
          continue WHEN _value IS null
                     OR _value IN ('null','','''');

          -- dbrid des Datensatzes ermitteln
          EXECUTE format('SELECT dbrid FROM %I WHERE %I = %L', _table, _field, _key)
            INTO _r_dbrid;

          -- Parameter anlegen falls noch nicht vorhanden
          IF NOT EXISTS( SELECT true
                           FROM recnokeyword
                          WHERE r_dbrid = _r_dbrid
                            AND r_reg_pname = _r.nexm_reg_pname ) THEN

            -- Where-Teil zum ergänzen falls definiert
            IF _r.nexm_where IS NOT null AND _r.nexm_where <> '' THEN
              _where = ' WHERE ' || _r.nexm_where;
            ELSE
              _where = '';
            END IF;

            RAISE NOTICE '_value: %', _value;

            -- Angaben zum Parameter ermitteln
            EXECUTE format('SELECT %L AS r_reg_pname,
                                   %L AS r_tablename,
                                   dbrid AS r_dbrid,
                                   r_descr,
                                   r_value,
                                   r_unit, r_min, r_max, coalesce(r_not,''false'') AS r_not, r_txt
                              FROM jsonb_to_record(%L) AS x(r_descr varchar, r_value varchar, r_unit varchar, r_min numeric, r_max numeric, r_not boolean, r_txt text)
                              JOIN %I ON %I = %L
                              %s'
                            , _r.nexm_reg_pname, _table, _value::jsonb, _table, _field, _key, _where) INTO _recno;

            -- Wenn kein Wert gefunden wurde ~> nichts schreiben
            continue WHEN _recno.r_value IS null
                       OR _recno.r_value IN ('null','','''');

            RAISE NOTICE 'r_value: %', _recno.r_value;

            -- Query für Anlage des Paramters erstellen und ausführen
            _query = format('INSERT INTO recnokeyword ( r_reg_pname, r_tablename, r_dbrid,
                                                        r_descr,
                                                        r_value,
                                                        r_unit, r_min, r_max, r_not, r_txt )
                                         VALUES       ( %L, %L, %L,
                                                        %L,
                                                        CAST(%s as %s),
                                                        %L, %L, %L, %L, %L )'
                            ,                           _recno.r_reg_pname, _recno.r_tablename, _recno.r_dbrid,
                                                        _recno.r_descr,
                                                        _recno.r_value, coalesce(_r.nexm_cast, 'varchar'),
                                                        _recno.r_unit, _recno.r_min, _recno.r_max, _recno.r_not, _recno.r_txt);

            RAISE NOTICE 'Parameter schreiben: %', _query;

            EXECUTE _query;

          END IF;

        END IF;

      END LOOP;

    END $$ LANGUAGE plpgsql;
--

-- Funktion schreibt Artikelstamdaten
CREATE OR REPLACE FUNCTION x_10_interfaces.nexar__import_art__by_jsonb(
      IN _part   jsonb,
      IN _ak_nr  varchar
    )
    RETURNS VOID AS $$
    BEGIN

      -- Pflichtfelder schreiben
      --- keine vorhanden

      -- dynamische Felder schreiben
      PERFORM x_10_interfaces.nexar__import_dynamic_fields__by_jsonb(_part, 'art', 'ak_nr', _ak_nr);

    END $$ LANGUAGE plpgsql;
  --

-- Funktion löst Unterfunktionen für mehrere Teile aus
CREATE OR REPLACE FUNCTION x_10_interfaces.nexar__import_multi__from_jsonb(
      IN _results           jsonb,
      IN _write_to_art      boolean DEFAULT false,
      IN _write_to_epreis   boolean DEFAULT false,
      IN _write_to_anfrage  boolean DEFAULT true
      )
    RETURNS VOID AS $$
    DECLARE
      _element           record;
      _ak_nr             varchar;
    BEGIN

      -- einzelne Objekte durchlaufen
      FOR _element IN
          SELECT jsonb_array_elements FROM jsonb_array_elements(_results)
      LOOP

        -- Artikel ermitteln
        SELECT _element.jsonb_array_elements ->> 'reference'
          INTO _ak_nr;
        -- Debug-Info
        RAISE NOTICE 'Artikelnummer: %', _ak_nr;

        -- Artikeldaten übernehmen
        IF _write_to_art THEN
          EXECUTE x_10_interfaces.nexar__import_art__by_jsonb( x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.parts[0]' ), _ak_nr );
        END IF;

        -- Lieferantenpreise übernehmen
        IF _write_to_epreis THEN
          EXECUTE x_10_interfaces.nexar__import_epreis__by_jsonb( x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.parts[0].sellers' ), _ak_nr );
        END IF;

        -- Anfrageangebot anlegen
        IF _write_to_anfrage THEN
          EXECUTE x_10_interfaces.nexar__import_anfangebot__by_jsonb( x_10_interfaces.jsonb__by_dynamic_json_path__get( _element.jsonb_array_elements, '$.parts[0].sellers' ), _ak_nr );
        END IF;

      END LOOP;

    END $$ LANGUAGE plpgsql;
  --

-- Funktion löst Unterfunktionen für ein Teil aus (Artikelstamm)
CREATE OR REPLACE FUNCTION x_10_interfaces.nexar__import__from_jsonb(
      IN _results           jsonb,
      IN _ak_nr             varchar,
      IN _write_to_art      boolean DEFAULT true,
      IN _write_to_epreis   boolean DEFAULT true,
      IN _write_to_anfrage  boolean DEFAULT true
    )
    RETURNS VOID AS $$
  DECLARE
    _mpn               varchar;
    _nexar_part_id     varchar;
  BEGIN

    -- Prüfung ob Artikelnummer mitgegeben wurde, falls nicht -> Artikelnummer anhand MPN suchen
    IF _ak_nr is null THEN
      -- MPN ermitteln
      SELECT x_10_interfaces.jsonb__by_dynamic_json_path__get( _results, '$.results[0].part' ) ->> 'mpn',
             x_10_interfaces.jsonb__by_dynamic_json_path__get( _results, '$.results[0].part' ) ->> 'id'
        INTO _mpn,
           _nexar_part_id;
    -- Artikelnummer suchen
      _ak_nr = ( SELECT ak_nr FROM art WHERE ak_herstelleraknr = _mpn LIMIT 1 );
    -- Debug Info
      RAISE NOTICE 'MPN: %, Part_ID: %, gefundener Artikel: %', _mpn, _nexar_part_id, _ak_nr;
    END IF;

    -- Artikeldaten übernehmen
    IF _write_to_art THEN
      EXECUTE x_10_interfaces.nexar__import_art__by_jsonb( x_10_interfaces.jsonb__by_dynamic_json_path__get( _results, '$.results[0].part' ), _ak_nr );
    END IF;

    -- Lieferantenpreise übernehmen
    IF _write_to_epreis THEN
      EXECUTE x_10_interfaces.nexar__import_epreis__by_jsonb( x_10_interfaces.jsonb__by_dynamic_json_path__get( _results, '$.results[0].part.sellers' ), _ak_nr );
    END IF;

    -- Anfrageangebot anlegen
    IF _write_to_anfrage THEN
      EXECUTE x_10_interfaces.nexar__import_anfangebot__by_jsonb( x_10_interfaces.jsonb__by_dynamic_json_path__get( _results, '$.results[0].part.sellers' ), _ak_nr );
    END IF;

    END $$ LANGUAGE plpgsql;
  --